package gov.va.med.mhv.getcare.validator;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.LinkedHashMap;
import java.util.regex.Pattern;

public final class Validations {
	private static final String EMAIL_PATTERN = "^[_A-Za-z0-9-\\+]+(\\.[_A-Za-z0-9-]+)*@[A-Za-z0-9-]+(\\.[A-Za-z0-9]+)*(\\.[A-Za-z]{2,})$";

	private static final String ZIPCODE_PATTERN1 = " *\\d{5} *"; // 5 digits
	private static final String ZIPCODE_PATTERN2 = " *\\d{9} *"; // 9 digits
	private static final String ZIPCODE_PATTERN3 = " *\\d{5}-\\d{4} *"; // 99999-9999

	private static final String PHONE_PATTERN1 = " *\\d{10} *"; // 10 digits
	private static final String PHONE_PATTERN2 = " *\\d{12} *"; // 12 digits
	private static final String PHONE_PATTERN3 = " *\\(\\d{3}\\) ?\\d{3}-\\d{4} *"; // (999)999-9999,
	private static final String PHONE_PATTERN4 = " *\\d{3}\\.\\d{3}\\.\\d{4} *"; // 999.999.9999
	private static final String PHONE_PATTERN5 = " *\\d{3}-\\d{3}-\\d{4} *"; // 999-999-9999

	public static void validateEmail(String fieldName, String value, LinkedHashMap<String, String> validationErrors) {

		Pattern pattern = Pattern.compile(EMAIL_PATTERN);
		if (!pattern.matcher(value).matches()) {
			validationErrors.put(fieldName, fieldName + " is invalid");
		}
	}

	public static void validateZipCode(String fieldName, String value, LinkedHashMap<String, String> validationErrors) {

		if (!value.matches(ZIPCODE_PATTERN1) && !value.matches(ZIPCODE_PATTERN2) && !value.matches(ZIPCODE_PATTERN3)) {
			validationErrors.put(fieldName, fieldName + " is invalid");
		}
	}

	public static void validatePhoneNumber(String fieldName, String value, LinkedHashMap<String, String> validationErrors) {

		if (!value.matches(PHONE_PATTERN1) && !value.matches(PHONE_PATTERN2) && !value.matches(PHONE_PATTERN3) && !value.matches(PHONE_PATTERN4)
				&& !value.matches(PHONE_PATTERN5)) {
			validationErrors.put(fieldName, fieldName + " is invalid");
		}
	}

	public static void validateMaximumLength(String fieldName, String value, int maxLength, boolean required, LinkedHashMap<String, String> validationErrors) {

		if (required) {
			if (!hasValue(value)) {
				validationErrors.put(fieldName, fieldName + " is required");
				return;
			}
		}

		if (hasValue(value)) {
			if (value.length() > maxLength) {
				validationErrors.put(fieldName, fieldName + " is too long");
			}
		}

	}

	public static boolean hasValue(String s) {
		return null != s && s.trim().length() > 0;
	}

	public static boolean hasValue(Object o) {
		return null != o;
	}

	public static void validateRequired(String fieldName, String value, LinkedHashMap<String, String> validationErrors) {
		if (!hasValue(value)) {
			validationErrors.put(fieldName, fieldName + " is required");
		}
	}

	public static void validateRequired(String fieldName, Object value, LinkedHashMap<String, String> validationErrors) {
		if (!hasValue(value)) {
			validationErrors.put(fieldName, fieldName + " is a required field");
		}
	}

	public static void validateRequired(String fieldName, String month, String day, String year, String property,
			LinkedHashMap<String, String> validationErrors) {
		if (!hasValue(day) || !hasValue(month) && !hasValue(year)) {
			validationErrors.put(fieldName, fieldName + " is a required field");
		}
	}

	public static void validateMinValue(String fieldName, String value, BigDecimal minValue, String property, LinkedHashMap<String, String> validationErrors) {
		if (isValidNumber(value)) {
			validateMinValue(fieldName, new BigDecimal(value), minValue, property, validationErrors);
		}
	}

	public static void validateMinValue(String fieldName, Comparable value, Comparable minValue, String property,
			LinkedHashMap<String, String> validationErrors) {
		if (value != null && value.compareTo(minValue) < 0) {
			validationErrors.put(fieldName, property + "value must be greater than or equal to  " + minValue);
		}
	}

	public static void validateMaxValue(String fieldName, String value, BigDecimal maxValue, String property, LinkedHashMap<String, String> validationErrors) {
		if (isValidNumber(value)) {
			validateMaxValue(fieldName, new BigDecimal(value), maxValue, property, validationErrors);
		}
	}

	public static void validateMaxValue(String fieldName, Comparable value, Comparable maxValue, String property,
			LinkedHashMap<String, String> validationErrors) {
		if (value != null && value.compareTo(maxValue) > 0) {
			validationErrors.put(fieldName, property + "value must be less than or equal to  " + maxValue);
		}
	}

	public static void validateMinLength(String fieldName, String value, int minLength, LinkedHashMap<String, String> validationErrors) {
		if (hasValue(value) && value.trim().length() < minLength) {
			// TODO: change the error message
			validationErrors.put(fieldName, "Miminum length not met for " + fieldName);
		}
	}

	public static void validateMaxLength(String fieldName, String value, int maxLength, LinkedHashMap<String, String> validationErrors) {
		if (hasValue(value) && value.trim().length() > maxLength) {
			// TODO: change the error message
			validationErrors.put(fieldName, "Maximum length  met for " + fieldName);
		}
	}

	public static void validateScale(String fieldName, String value, int scale, String property, LinkedHashMap<String, String> validationErrors) {
		if (isValidNumber(value)) {
			validateScale(fieldName, new BigDecimal(value), scale, property, validationErrors);
		}
	}

	public static void validateScale(String fieldName, BigDecimal value, int scale, String property, LinkedHashMap<String, String> validationErrors) {
		if (value != null && value.scale() > scale) {
			validationErrors.put(fieldName, property + "value must contain at most " + scale + " numbers after the decimal");
		}
	}

	public static void validateEnumeration(String fieldName, String value, Enum[] enums, LinkedHashMap<String, String> validationErrors) {
		boolean valid = false;
		if (value != null) {
			for (int i = 0; i < enums.length; i++) {
				if (value.equals(enums[i].toString())) {
					valid = true;
					break;
				}
			}
		}
		if (!valid) {
			validationErrors.put(fieldName, fieldName + " is not Valid");
		}
	}

	public static void validateInteger(String fieldName, String value, LinkedHashMap<String, String> validationErrors) {
		if (hasValue(value) && !Validations.isValidInteger(value)) {
			// messages.addMessage(MessageUtils.createErrorMessage(CoreMessages.INVALID_NUMBER,
			// new String [] {property}, new Object[]{ value }));
			validationErrors.put(fieldName, "Invalid Number for " + fieldName);
		}
	}

	public static void validateLong(String fieldName, String value, LinkedHashMap<String, String> validationErrors) {
		if (hasValue(value) && !Validations.isValidLong(value)) {
			// messages.addMessage(MessageUtils.createErrorMessage(CoreMessages.INVALID_NUMBER,
			// new String [] {property}, new Object[]{ value }));
			validationErrors.put(fieldName, "Invalid Number for " + fieldName);
		}
	}

	public static final void validateBoolean(String fieldName, String value, LinkedHashMap<String, String> validationErrors) {
		// TODO Implement this. For now, can leave blank, as unknown strings
		// will become false Booleans, which is fine
	}

	public static void validateBigDecimal(String fieldName, String value, LinkedHashMap<String, String> validationErrors) {
		if (hasValue(value) && !Validations.isValidDecimal(value)) {
			validationErrors.put(fieldName, "Invalid Number for " + fieldName);
			// messages.addMessage(MessageUtils.createErrorMessage(CoreMessages.INVALID_NUMBER,
			// new String [] {property}, new Object[]{ value }));
		}
	}

	public static void validateDate(String fieldName, String month, String day, String year, boolean dayRequired,
			LinkedHashMap<String, String> validationErrors) {
		if (!Validations.isValidDate(month, day, year, dayRequired)) {
			validationErrors.put(fieldName, "Invalid Date for " + fieldName);
			// messages.addMessage(MessageUtils.createErrorMessage(CoreMessages.INVALID_DATE,
			// new String [] {property}, new Object[]{}));
		}
	}

	public static void validateTimestamp(String fieldName, String month, String day, String year, String hours, String minutes, boolean dayRequired,
			LinkedHashMap<String, String> validationErrors) {
		if (!Validations.isValidTimestamp(month, day, year, hours, minutes, dayRequired)) {
			validationErrors.put(fieldName, "Invalid Date for " + fieldName);
			// messages.addMessage(MessageUtils.createErrorMessage(CoreMessages.INVALID_DATE,
			// new String [] {property}, new Object[]{}));
		}
	}

	private static boolean isValidNumber(String s) {
		try {
			BigDecimal dec = new BigDecimal(s);
			return dec != null;
		} catch (Exception ex) {
			return false;
		}
	}

	private static boolean isValidInteger(String value) {
		try {
			Integer.parseInt(value);
			return true;
		} catch (NumberFormatException nfe) {
			return false;
		}
	}

	private static boolean isValidDate(String month, String day, String year, boolean dayRequired) {
		// A blank date may or may not be valid - up the the BO
		if (!hasValue(day) || !hasValue(month) && !hasValue(year)) {
			return true;
		}

		try {
			int iMonth = Integer.parseInt(month) - 1;
			int iDay;
			// Only default the day if it is not required AND blank.
			// In any other case try to parse it
			if (!dayRequired && !hasValue(day)) {
				iDay = 1;
			} else {
				iDay = Integer.parseInt(day);
			}
			int iYear = Integer.parseInt(year);

			Calendar cal = GregorianCalendar.getInstance();
			cal.clear();
			cal.setLenient(false);
			cal.set(Calendar.MONTH, iMonth);
			cal.set(Calendar.DAY_OF_MONTH, iDay);
			cal.set(Calendar.YEAR, iYear);

			// This will cause an exception if the day is invalid (e.g.
			// 2/29/2001)
			cal.getTime();

			return true;
		} catch (Exception ex) {
			return false;
		}
	}

	private static boolean isValidTimestamp(String month, String day, String year, String hours, String minutes, boolean dayRequired) {
		try {
			boolean validDate = isValidDate(month, day, year, dayRequired);
			// TODO Handle time portion
			return validDate;
		} catch (Exception pe) {
			return false;
		}
	}

	private static boolean isValidLong(String value) {
		try {
			Long.parseLong(value);
			return true;
		} catch (NumberFormatException nfe) {
			return false;
		}
	}

	private static boolean isValidDecimal(String value) {
		try {
			new BigDecimal(value);
			return true;
		} catch (NumberFormatException nfe) {
			return false;
		}
	}

}
